using System;
using System.Data;
using System.Collections;
using System.Reflection;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for PatientUnitSelection.
    /// 
	/// </summary>
	public class PatientUnitSelection : BaseBusinessObject
	{

		#region variables
		/// <summary>
		/// Current use who is selecting units
		/// </summary>
		private BOL.VbecsUser _thisUser;
        /// <summary>
        /// Patient that units are being selected against
        /// </summary>
		private Patient _patient;
		/// <summary>
		/// Collection class that holds selected units
		/// </summary>
		private SelectedUnits _selectedUnits;
		/// <summary>
		/// Ordered component that units are being selected against
		/// </summary>
		private OrderedComponent _orderedComponent;
		/// <summary>
		/// Boolean indicating if unit is eligible for electronic crossmatch
		/// </summary>
		private bool _bloodUnitEligibleForEXM;
		/// <summary>
		/// Holds validation messages associated with the selection of 1 blood unit
		/// </summary>
		private ArrayList _validationMessages;
		/// <summary>
		/// Comments associated with the Unit Selection
		/// </summary>
		private ArrayList _orderedComponentComments;
		/// <summary>
		/// Boolean indicating if the facility is full service
		/// </summary>
		private bool _isFullService;
		/// <summary>
		/// Boolean indicating if units must be selected under emergency issue
		/// </summary>
		private bool _isEmergencyIssue;
		/// <summary>
		/// String indicating the reason this order must be emergency issued
		/// </summary>
		private string _emergencyIssueReason;
		/// <summary>
		/// Collection containing all of the units that the user has decided need
		/// an optional crossmatch
		/// </summary>
		private ArrayList _optionalCrossmatchUnits;
		/// <summary>
		/// Boolean indicating if the order had a specimen association prior to 
		/// processing in this UC.  The is necessary for the Optional XM
		/// functionality.
		/// </summary>
		private bool _previousSpecimenAssociation = false;
		/// <summary>
		/// Under emegency issue situations, an order could have been accepted 
		/// without a specimen.  The specimen can then be identified at a later time
		/// which is what this indicates.
		/// </summary>
		private bool _specimenIdentified;

		#endregion


		#region constructors

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3387"> 
		///		<ExpectedInput>Valid OrderedComponent and VbecsUser</ExpectedInput>
		///		<ExpectedOutput>Instance of PatientUnitSelection</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3388"> 
		///		<ExpectedInput>Null OrderedComponent</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="4002"> 
		///		<ExpectedInput>Null VbecsUser</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Since multiple Ordered Components can be accepted for an order, we need to accept a 
		/// collection.  We must also accept an ArrayList of Ordered Tests.  We'll act as a messenger
		/// and pass them on to UC56. 
		/// </summary>
		public PatientUnitSelection(BOL.OrderedComponent orderedComponent, BOL.VbecsUser thisUser)
		{
			if (orderedComponent == null || thisUser == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("OrderedComponent, VbecsUser").ResString);
			}

			_orderedComponent = orderedComponent;
	
			_thisUser = thisUser;
			_isFullService = new Division(_orderedComponent.DivisionCode).ServiceTypeIndicator;

			_patient = orderedComponent.Patient;
			_optionalCrossmatchUnits = new ArrayList();

			// set here so it's not called every time the property is called
			_isEmergencyIssue = GetEmergencyIssueStatus();

			_selectedUnits = new SelectedUnits(this);

			// This holds of the the messages (BOL.PatientUnitSelectionMessgage) that result when rules 
			// are applied to Blood Units
			_validationMessages = new ArrayList();
			_orderedComponentComments = new ArrayList();
		
			

			if (_orderedComponent.SpecimenIdentified) // CR3114
			{
				_previousSpecimenAssociation = true;
			}
		}

		#endregion


		#region properties

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6594"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6595"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SpecimenIdentified
		/// </summary>
		public bool SpecimenIdentified
		{
			get
			{
				return _specimenIdentified;
			}
			set
			{
				_specimenIdentified = value;
			}
		}


		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6598"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6599"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns units that have been ordered in previous sessions
		/// </summary>
		/// <returns></returns>
		public DataTable AlreadyOrderedUnits
		{
			get
			{
				return _selectedUnits.AlreadySelectedUnits;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5038"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5039"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean value indicating whether units must be selected
		/// under emergency issue.
		/// </summary>
		public bool IsEmergencyIssue
		{
			get
			{
				return _isEmergencyIssue;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5040"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>A string that varies depending on emergency issue status</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5041"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the reason why the units must be emergency issued.
		/// </summary>
		public string EmergencyIssueReason
		{
			get
			{
				// _emergencyIssueReason is set at the same time _isEmergencyIssue 
				// is set (when GetEmergencyIssueStatus is called by the constructor)
				if (_emergencyIssueReason != null) return _emergencyIssueReason;
				else return string.Empty;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3417"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>A valid Ordered Component</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="3419"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This represents the Ordered Component that the user is selecting
		/// units for.
		///</summary>
		public BOL.OrderedComponent OrderedComponent
		{
			get 
			{
				return _orderedComponent;
			}
			set 
			{
				_orderedComponent = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3423"> 
		///		<ExpectedInput>A valid SelectedUnits object</ExpectedInput>
		///		<ExpectedOutput>A valid SelectedUnits object</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="1415"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This represents the units chosen for the order.
		/// </summary>
		public SelectedUnits SelectedUnitsCollection
		{
			get 
			{
				return _selectedUnits;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3425"> 
		///		<ExpectedInput>A valid DataTable of directed blood units</ExpectedInput>
		///		<ExpectedOutput>A valid DataTable of directed blood units</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1410"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This returns blood units that are directed to a patient. This is 
		/// a pass-thru method to the inner class.
		///</summary>
		public DataTable DirectedBloodUnits
		{
			get
			{
				return _selectedUnits.DirectedUnits;
			}
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3429"> 
		///		<ExpectedInput>An ArrayList of Validation Messages</ExpectedInput>
		///		<ExpectedOutput>An ArrayList of Validation Messages</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="1416"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This represents all of the error and warning messages associated with
		/// the selection of one unit.
		///</summary>
		public ArrayList ValidationMessages
		{
			get
			{
				return _validationMessages;
			}
			set
			{
				_validationMessages = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/3/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3431"> 
		///		<ExpectedInput>An ArrayList of OrderedComponentComments</ExpectedInput>
		///		<ExpectedOutput>An ArrayList of OrderedComponentComments</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="1414"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This represents all of the comments associated with the order.
		///</summary>
		public ArrayList OrderedComponentComments
		{
			get
			{
				return _orderedComponentComments;
			}
			set
			{
				_orderedComponentComments = value;
			}
		}


		#endregion


		#region methods

		#region optional crossmatch

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5913"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>None </ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5914"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds a crossmatch test for a unit in the Component class of OTHER
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void AddOptionalCrossmatch(Guid bloodUnitGuid)
		{
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
			}

			_optionalCrossmatchUnits.Add(bloodUnitGuid);
			RuleBroken("OptionalCrossmatchUnits", _optionalCrossmatchUnits.Count>0 && _orderedComponent.SpecimenIdentified == false);  // CR3114
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7164"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7165"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Removes a crossmatch test for a unit in the Component class of OTHER
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public void RemoveOptionalCrossmatch(Guid bloodUnitGuid)
		{
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
			}

			if (_optionalCrossmatchUnits.Contains(bloodUnitGuid))
			{
				_optionalCrossmatchUnits.Remove(bloodUnitGuid);
			}
			RuleBroken("OptionalCrossmatchUnits", _optionalCrossmatchUnits.Count>0 && _orderedComponent.SpecimenIdentified == false);  // CR3114
			
			if (_optionalCrossmatchUnits.Count == 0 && !_previousSpecimenAssociation)
			{
				_orderedComponent.AssociateWithSpecimen(null);
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7358"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7359"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>Indicates if an optional crossmatch was ordered
		///	on any o</summary>
		public bool OptionalCrossmatchOrdered()
		{
			return _optionalCrossmatchUnits.Count > 0;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5915"> 
		///		<ExpectedInput>Guid from a unit that was previously ordered</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5916"> 
		///		<ExpectedInput>New Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean indicating if an optional crossmatch test
		/// was ordered for the unit. (Needed for BR_3.09)
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public bool OptionalCrossmatchOrdered(Guid bloodUnitGuid)
		{
			return (_optionalCrossmatchUnits.Contains(bloodUnitGuid));
		}

		#endregion

		
		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8106"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8107"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ClearClassObjects
		/// </summary>
		public void ClearClassObjects()
		{
			this._thisUser.Dispose();
			this._patient.AboRh.Dispose();
			this._patient.Dispose();
			//
			this._selectedUnits = null;
			//
			this._orderedComponent.Dispose();
			//
			for (int idx = 0; idx < this._validationMessages.Count; idx++)
			{	
				((PatientUnitSelectionMessage)this._validationMessages[idx]).Dispose();
			}
			//
			this. _orderedComponentComments = null;
			this._emergencyIssueReason = null;
			this._optionalCrossmatchUnits = null;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7603"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7604"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AllAutologousUnitsChosen
		/// </summary>
		/// <returns></returns>
		public bool AllAutologousUnitsChosen()
		{
			return _selectedUnits.AllAutologousUnitsChosen(Guid.Empty);		
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7605"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7606"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AllDirectedUnitsChosen
		/// </summary>
		/// <returns></returns>
		public bool AllDirectedUnitsChosen()
		{
			return _selectedUnits.AllDirectedUnitsChosen(Guid.Empty);			
		}

		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8347"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8348"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AllRestrictedUnitsChosen
		/// </summary>
		/// <returns></returns>
		public bool AllRestrictedUnitsChosen()
		{
			return _selectedUnits.AllRestrictedUnitsChosen(Guid.Empty);			
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7607"> 
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7608"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AlreadySelectedInPreviousSession
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public bool AlreadySelectedInPreviousSession(Guid bloodUnitGuid)
		{
			if ( (this.AlreadyOrderedUnits.Select(TABLE.OrderedUnit.BloodUnitGuid + " = '" + bloodUnitGuid + "'")).Length > 0)
			{
				return true;
			}
			else 
			{
				return false;
			}

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7609"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7610"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a bool value indicating whether or not this is an emergency issue condition.
		///  CR2789: Code that checked for ConversionDiscrepancy was removed from first "if" statement:
		///			_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy
		/// </summary>
		/// <returns></returns>
		public bool GetEmergencyIssueStatus()
		{
			// BR_3.05, TT_52.03: Patient must have specimen test results and ABO/Rh may not 
			// be inconclusive.
			Common.ABORhSource aboRhSource = _patient.ABORhSource;

			if (_patient.ABORhSource == Common.ABORhSource.Conversion
				|| _patient.ABORhSource == Common.ABORhSource.None
				|| _patient.AboRh.Abo == Common.ABO.I 
				|| _patient.AboRh.RH == Common.RH.Inconclusive)
			{
				_emergencyIssueReason = Common.StrRes.ValidMsg.UC003.BR305NoABORhResults().ResString;
				return true;
			}

			if (this.OrderedComponent.SpecimenRequiredIndicator)
			{
				// Calling BR_3.01
				if (!ValidSpecimenExists())
				{
					_emergencyIssueReason = Common.StrRes.ValidMsg.UC003.BR301Specimen().ResString;
					return true;
				}
				else
				{
					//CR2789 start
					//BR_3.30
					if (_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy || 
						_patient.ABORhSource == Common.ABORhSource.SpecimenTestDiscrepancy)
					{
						_emergencyIssueReason = Common.StrRes.ValidMsg.UC003.BR330SpecimenResults().ResString;
						return true;
					}
						//BR_3.40
					else if (_orderedComponent.GetSpecimen().RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy)  // CR3114
					{
						_emergencyIssueReason = Common.StrRes.ValidMsg.UC003.BR340SpecimenResults().ResString;
						return true;
					}
					//CR2789 end	
				}

				// BR_3.24
				if (_patient.JustifiedAboRH)
				{
					if (!_orderedComponent.GetSpecimen().ABORhTested)  // CR3114
					{
						string reason = _patient.GetPatientAboRhJustificationReason();
						_emergencyIssueReason = Common.StrRes.ValidMsg.UC003.BR324ABORhConfirmation(reason).ResString;
						return true;
					}
				}
			}
			return false;
		}

		#region authority

		///<summary>
		/// Checks that the user has authority to issue blood that is not ABO compatible.
		/// BR_6.02
		///</summary>
		private bool HasAuthorityToIssueABOIncompatibleBlood()
		{
			return _thisUser.IsAuthorizedForFunction(Common.FunctionCode.SelectIssueABOIncompatibleblood);
		}

		///<summary>
		/// Checks that the user has authority to issue blood that is antigen positive
		/// or has not been typed yet.
		///</summary>
		private bool HasAuthorityToIssueAntigenPositiveOrUntypedRBCs()
		{
			return _thisUser.IsAuthorizedForFunction(Common.FunctionCode.IssueAntigenPosOrUntypedRedbloodcells);
		}

		/// <summary>
		/// Checks that the user has authority to issue units that are expired.
		/// </summary>
		/// <returns></returns>
		private bool HasAuthorityToIssueExpiredUnits()
		{
			return _thisUser.IsAuthorizedForFunction(Common.FunctionCode.IssueExpiredBloodProducts);
		}

		#endregion


		#region methods used to populate UI elements

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3520"> 
		///		<ExpectedInput>DivisionCode - where everything is valid</ExpectedInput>
		///		<ExpectedOutput>EXM is possible</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3521"> 
		///		<ExpectedInput>DivisionCode - with Test discrepancy</ExpectedInput>
		///		<ExpectedOutput>EXM not enabled</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="3434"> 
		///		<ExpectedInput>DivisionCode - with Justified ABO</ExpectedInput>
		///		<ExpectedOutput>EXM not enabled</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3435"> 
		///		<ExpectedInput>A null division code</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="7183"> 
		///		<ExpectedInput>A division where eXM is not enabled</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating EXM isn't enabled</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="7251"> 
		///		<ExpectedInput>DivisionCode - with No specimen</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating no specimen</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="5044"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating no specimen results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5045"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating positive antibody results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5046"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating antibody history</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8846"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating antibody history (this one is from conversion)</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5047"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating Antigen Negative transfusion requirement</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5048"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating typing difficulty</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="7252"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating typing difficulty</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5049"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating ABO/Rh history discrepancy</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7253"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating ABO/Rh history discrepancy</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5050"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Warning message indicating ABO/Rh history discrepancy</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This returns string that is used to populate txtEXMStatus.
		/// BR_3.17, BR_3.18
		///</summary>
		/// <param name="division"></param>
		/// <returns></returns>
		public string GetDivisionAndPatientEXMStatus(BOL.Division division)
		{
			if (division == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Division").ResString);
			}

			string status = String.Empty;
			if (division.ElectronicCrossmatchIndicator == true) 
			{
				// The patient part of TT_3.05 is implemented in the following method calls			
				return _patient.DetermineElectronicCrossmatchEligibility(this.OrderedComponent.SpecimenIdentified ? this.OrderedComponent.GetSpecimen() : null, _orderedComponent.DivisionCode);  // CR3114
			}
			else
			{
				return Common.StrRes.InfoMsg.Common.UnitEXMDisabled().ResString;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3438"> 
		///		<ExpectedInput>A blood unit that will qualify for eXM</ExpectedInput>
		///		<ExpectedOutput>A string indicating that the blood unit qualifies for eXM</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3439"> 
		///		<ExpectedInput>A blood unit that will not qualify for eXM</ExpectedInput>
		///		<ExpectedOutput>A string indicating that the blood unit doesn't qualify for eXM along with the reason why</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///	This method returns a string indicating if the unit is eliigble for electronic crossmatch (eXM).
		///	If the unit doesn't qualify for eXM, the reason is given.
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <returns></returns>
		public string GetUnitEXMStatusText(BOL.BloodUnit bloodUnit)
		{
			return DetermineUnitEXM(bloodUnit);
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/3/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5921"> 
		///		<ExpectedInput>Valid unit</ExpectedInput>
		///		<ExpectedOutput>string representing time</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5922"> 
		///		<ExpectedInput>null unit</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This calculates the amount of time until a unit expires.
		/// </summary>
		/// BR_3.22
		public string GetTimeUntilExpiration(BOL.BloodUnit bloodUnit)
		{
			if (bloodUnit == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnit").ResString);
			}

			DateTime expiration = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			
			if (BOL.VBECSDateTime.GetDivisionCurrentDateTime() > expiration) return "Expired";

			string expires = String.Empty; 
		
			System.TimeSpan timeSpan =  expiration.Subtract(BOL.VBECSDateTime.GetDivisionCurrentDateTime());
			int days	= timeSpan.Days;
			int hours	= timeSpan.Hours;
			int minutes = timeSpan.Minutes;
			
			if (days > 1) return days + " days";
			else if (days == 1) return days + " day";
			else if (hours > 1) return hours + " hours";
			else if (hours == 1) return hours + " hour";
			else return minutes + " minutes";
		}

		#endregion

	
		
		///<summary>
		/// This method serves two purposes.  First, it returns a string that is displayed in UC03 and
		/// lets the user know the eXM status of the unit.  It also sets a bool value to indicate
		/// whether or not the unit is eligible for eXM.
		///</summary>
		///
		// BRs 3.17, 3.18, 3.35, 3.36
		private string DetermineUnitEXM(BOL.BloodUnit bloodUnit)
		{		
			_bloodUnitEligibleForEXM = false;

			// TT_3.05, 8: Selected unit must be ABO compatible based on TT_3.01 or TT_3.02.
			Common.ComponentClass componentClass = (Common.ComponentClass) bloodUnit.BloodComponentClassId;
			if (componentClass == Common.ComponentClass.WB || componentClass == Common.ComponentClass.RBC)
			{
				_validationMessages.Clear();

				switch(componentClass)
				{
					case Common.ComponentClass.WB:
					{
						this.ValidateWholeBloodTT301(bloodUnit);
						break;
					}
					case Common.ComponentClass.RBC:
					{
						ValidateNonWholeBloodNonPlasmaTT302(bloodUnit);
						break;
					}
				}
				if (_validationMessages.Count > 0)
				{
					return Common.StrRes.InfoMsg.Common.UnitEXMDisabledABOIncompatible().ResString;
				}
			}
			// TT_3.05, 9: Selected Rh negative units must have ABO and Rh confirmation results entered. 
			// The ABO and Rh confirmation interpretation must match the unit ABO/Rh at log-in.	
			// Selected Rh positive units must have ABO confirmation results entered. The ABO 
			// confirmation interpretation must match the unit ABO at log-in.	Unit not eligible
			// for eXM. ABO confirmation not performed.
			DataTable dtTests = BOL.BloodUnitTest.GetBloodUnitTests(bloodUnit.BloodUnitGuid, _orderedComponent.DivisionCode);
			DataRow [] drABOInterpretations = dtTests.Select(TABLE.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterp);
			DataRow [] drRhInterpretations = dtTests.Select(TABLE.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.RhInterp);
			
			if (drABOInterpretations.Length == 0)
			{
				return Common.StrRes.InfoMsg.Common.UnitEXMDisabledNoABORhConfirm().ResString;
			}
			else
			{
				foreach (DataRow dr in drABOInterpretations)
				{
					if (bloodUnit.BloodUnitMedia.AboRh.Abo != AboRh.GetAboFromString(dr[TABLE.TestResult.TestResultId].ToString()))
					{
						return Common.StrRes.InfoMsg.Common.UnitEXMDisabledABOConfirmNoMatch().ResString;
					}
				}

				// Unit is Rh negative
				if (bloodUnit.BloodUnitMedia.AboRh.RH == Common.RH.Negative)
				{
					
					foreach (DataRow dr in drRhInterpretations)
					{
						if (bloodUnit.BloodUnitMedia.AboRh.RH != AboRh.GetRHFromCode(dr[TABLE.TestResult.TestResultId].ToString()))
						{
							return Common.StrRes.InfoMsg.Common.UnitEXMDisabledRhConfirmNoMatch().ResString;
						}
					}
				}
			}
			_bloodUnitEligibleForEXM = true;
			return Common.StrRes.InfoMsg.Common.UnitEXMEnabled().ResString;
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3391"> 
		///		<ExpectedInput>A bloodunit that will qualify for electronic crossmatch (eXM)</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3392"> 
		///		<ExpectedInput>A bloodunit that will not qualify for eXM</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///This method returns a boolean value that indicates whether or not the unit is eligible for eXM.
		///</summary>
		public bool GetUnitEXMStatus( BOL.BloodUnit bloodUnit )
		{
			DetermineUnitEXM( bloodUnit );
			return _bloodUnitEligibleForEXM;
		}

	
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3393"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3394"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This method returns a boolean value if a valid specimen exists for the patient.
		/// A valid specimen is one that is acceptable. 
		/// BR_3.01
		///</summary>
		public bool ValidSpecimenExists()
		{
			// CR3114 start
			return ( _orderedComponent.SpecimenIdentified &&
				!_orderedComponent.GetSpecimen().IsNew &&  
				!_orderedComponent.GetSpecimen().UnacceptableSpecimenIndicator &&
				_orderedComponent.GetSpecimen().MaintainableSpecimenIndicator);
			// CR3114 end
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3397"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable containing valid blood units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3398"> 
		///		<ExpectedInput>Restricted Unit for other patient</ExpectedInput>
		///		<ExpectedOutput>DataTable NOT containing restricted unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7249"> 
		///		<ExpectedInput>RBC unit that doesn't qualify for EI</ExpectedInput>
		///		<ExpectedOutput>DataTable NOT containing this unit</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7250"> 
		///		<ExpectedInput>FFP unit that doesn't qualify for EI</ExpectedInput>
		///		<ExpectedOutput>DataTable NOT containing this unit</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// This returns a DataTable representing all units that meet compatibility requirements for an 
		/// order.
		/// Implements BR_BR_3.04, BR_3.13, part of 3.19
		///</summary>
		public DataTable GetCompatibleUnitsForDisplay()
		{           
			Common.ComponentClass componentClass = (Common.ComponentClass) _orderedComponent.ComponentOrdered.BloodComponentClassId;
			System.Data.DataTable dtRetVal = new System.Data.DataTable();
			switch(componentClass)
			{
					// If they need whole blood units  TT_3.01
					// 3.2.4.5.13
					// 3.2.4.5.15  This BR is implented in the stored proc for this method
				case Common.ComponentClass.WB:
					dtRetVal = DAL.PatientUnitSelection.GetCompatibleWholeBloodUnits(_patient.AboRh.Abo.ToString(), _patient.AboRh.RHFactorCode, _patient.PatientGuid, _orderedComponent.DivisionCode); 
					// Now, get any RBCs
					dtRetVal = GetOtherUnits(true, dtRetVal);
					break;
					// TT_3.02 If they need Non-Whole Blood, Non-Plasma ICCBA Component Classes
					// talked to Claudette and she is thinking of taking Other out of TT 3.02
					// at the time of building this list.
				case Common.ComponentClass.RBC: 
					dtRetVal = DAL.PatientUnitSelection.GetCompatibleNonWholeBloodNonPlasmaUnits(_patient.AboRh.Abo.ToString(), _patient.AboRh.RHFactorCode, _orderedComponent.DivisionCode);
					if (!_isEmergencyIssue)
					{
						dtRetVal = GetOtherUnits(false, dtRetVal);
					}
					break;
					// TT_3.04 ABO Compatibility Requirements for ICCBA Plasma-Only Component Classes
				case Common.ComponentClass.FFP:
					dtRetVal = DAL.PatientUnitSelection.GetCompatibleFFPUnits(_patient.AboRh.Abo.ToString(), _orderedComponent.DivisionCode); 
					break;
					// TT_3.04 ABO Compatibility Requirements for ICCBA Plasma-Only Component Classes
				case Common.ComponentClass.CRYO:
					dtRetVal = DAL.PatientUnitSelection.GetCompatibleCryoUnits(_patient.AboRh.Abo.ToString(), _orderedComponent.DivisionCode); 
					break;
				case Common.ComponentClass.PLT:
					dtRetVal = DAL.PatientUnitSelection.GetCompatiblePlateletBloodUnits(_orderedComponent.DivisionCode);
					break;
				case Common.ComponentClass.Other:
					dtRetVal = DAL.PatientUnitSelection.GetCompatibleOtherBloodUnits(_orderedComponent.DivisionCode);
					break;
			}
			
			//BR_3.13 and part of 3.19
			if (dtRetVal != null && dtRetVal.Rows.Count > 0)
			{
				DataTable dtOrderedUnits = BOL.OrderedUnit.GetOrderedUnitsForPatient(_patient.PatientGuid);
				ArrayList rowsForRemoval = new ArrayList();
				foreach(System.Data.DataRow dtRow in dtRetVal.Rows)
				{
					// Part of 3_19 that states:
					// Not be restricted for a different patient(autologous or directed)
					// Also the part that says unit not already assigned or crossmatched to the
					// patient same division and the user, current location is in the blood bank are taken care of in the sprocs
					// also no quarantine indicator and future expiration date is taken care of in the sprocs 

					bool bAlreadyCrossMatched = this.AlreadyCrossMatched((System.Guid) dtRow[TABLE.BloodUnit.BloodUnitGuid], (string) dtRow[TABLE.BloodUnit.EyeReadableUnitId]);	
					char donationTypeCode = ((string) dtRow[TABLE.DonationType.DonationTypeCode]).ToCharArray()[0];
					Common.UnitStatusCode unitStatusCode = Common.Utility.GetUnitStatusCodeFromString( (string) dtRow[TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode] );
			
					System.Guid patientGUID = Guid.Empty; 
					if (dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.DirectedPatientGuid) && dtRow[TABLE.BloodUnitMedia.DirectedPatientGuid] != DBNull.Value)
					{
						patientGUID = (System.Guid) dtRow[TABLE.BloodUnitMedia.DirectedPatientGuid];
					}

					if (unitStatusCode == Common.UnitStatusCode.Assigned || unitStatusCode == Common.UnitStatusCode.CrossmatchedAssigned)
					{
						bool found = false;
						foreach (DataRow dr in dtOrderedUnits.Rows)
						{
							if ( (Guid) dr[TABLE.BloodUnit.BloodUnitGuid] == (Guid) dtRow[TABLE.BloodUnit.BloodUnitGuid])
							{
								found = true;
								rowsForRemoval.Add(dtRow);
								break;
							}
						}
						if (found) continue;
					}
					
					if(IsRestrictedForDifferentPatient((string) dtRow[TABLE.BloodUnit.EyeReadableUnitId], donationTypeCode, patientGUID)
						|| bAlreadyCrossMatched)
					{	
						rowsForRemoval.Add(dtRow);
						continue;
					}

					// BR_3.04, TT_3.02: If emergency issue and RBC or Plasma, blood type must be O or AB respectively
					if (_isEmergencyIssue)
					{
						int componentClassId = (int) dtRow[TABLE.ComponentClass.ComponentClassId];
						Common.ABO abo =  Common.Utility.GetAboFromString( dtRow[TABLE.BloodUnitMedia.BloodTypeCode].ToString() );
						// RBC
						if (componentClassId == (int) Common.ComponentClass.RBC && abo != Common.ABO.O)
						{
							rowsForRemoval.Add(dtRow);
							continue;
						}
						// FFP
						if (componentClassId == (int) Common.ComponentClass.FFP && abo != Common.ABO.AB)
						{
							rowsForRemoval.Add(dtRow);
							continue;
						}
					}
				}

				foreach(DataRow dr in rowsForRemoval)
				{
					dtRetVal.Rows.Remove(dr);
				}
			}
			return dtRetVal;
		}

		private DataTable GetOtherUnits(bool wb, DataTable dtUnits)
		{
			DataTable dtNewUnits = null;
			if (wb)
			{
				dtNewUnits = DAL.PatientUnitSelection.GetCompatibleNonWholeBloodNonPlasmaUnits(_patient.AboRh.Abo.ToString(), _patient.AboRh.RHFactorCode, _orderedComponent.DivisionCode);
			}
			else
			{
				dtNewUnits = DAL.PatientUnitSelection.GetCompatibleWholeBloodUnits(_patient.AboRh.Abo.ToString(), _patient.AboRh.RHFactorCode, _patient.PatientGuid, _orderedComponent.DivisionCode); 
			}
			// merge datatables
			if (dtNewUnits.Rows.Count == 0) return dtUnits;
			else if (dtUnits.Rows.Count == 0) return dtNewUnits;
			else
			{
				for (int i=0; i<dtNewUnits.Rows.Count; i++)
				{
					DataRow drNewRow = dtUnits.NewRow();
					foreach (System.Data.DataColumn col in drNewRow.Table.Columns)
					{
						if (dtNewUnits.Columns.Contains(col.ColumnName))
						{
							drNewRow[col.ColumnName] = dtNewUnits.Rows[i][col.ColumnName];
						}
					}
					dtUnits.Rows.Add(drNewRow);

					//drNewRow.ItemArray = dtNewUnits.Rows[0].ItemArray;

					//dtUnits.Rows.Add(dtNewUnits.Rows[i].ItemArray);
				}
				return dtUnits;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/23/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7174"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7175"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This returns a bool value that indicates whether a blood unit is part of an outgoing 
		/// shipment.
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
		// BR_3.45
		public bool IsOnPendingOutgoingShipmentList(BOL.BloodUnit bu)
		{
			if (DAL.BloodUnitFinance.IsBloodUnitOnPendingOutgoingShipmentList(bu.BloodUnitGuid))
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR345OutgoingShipment().ResString;
				_validationMessages.Add( new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR345OutgoingShipment,false,messageText));
				return true;
			}
			return false;
		}

		/// <summary>
		/// This returns a bool value that indicates whether a blood unit is already restricted 
		/// for another individual.
		/// </summary>
		/// <param name="eyeReadableUnitID"></param>
		/// <param name="donationTypeCode"></param>
		/// <param name="directedPatientGuid"></param>
		/// <returns></returns>
		// BR_3.46
		private bool IsRestrictedForDifferentPatient(string eyeReadableUnitID, char donationTypeCode, System.Guid directedPatientGuid)
		{
			// Commented out the autologous/directed logic because an allogeneic unit can be restricted in a
			// transfusion only facility! - C. Jensen 20040923
			if(		directedPatientGuid != Guid.Empty && 
				directedPatientGuid != _patient.PatientGuid)
			{			
				string messageText = Common.StrRes.ValidMsg.UC003.BR346UnitRestricted().ResString;
				_validationMessages.Add( new PatientUnitSelectionMessage(eyeReadableUnitID, Common.UC03UnitSelectErrorMessages.BR346UnitRestricted,false,messageText));
				return true;
			}
			else
			{
				return false;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3401"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3402"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Returns a boolean indicating whether or not the optional crossmatch
		/// feature should be enabled.
		///</summary>
		// BR_3.09/ TT_3.07
		public bool DisplayOptionalCrossmatch(BOL.BloodUnit bloodUnit)
		{
			Common.ProductType productTypeCode = Common.Utility.GetProductTypeFromProductTypeCode(bloodUnit.ProductType.ProductTypeCode);
			if (productTypeCode == Common.ProductType.ApheresisPlasma || // E014
				productTypeCode == Common.ProductType.ThawedApheresisPlasma || // E015
				productTypeCode == Common.ProductType.Plasma || // E017
				productTypeCode == Common.ProductType.ThawedPlasma || // E018
				productTypeCode == Common.ProductType.PlateletRichPlasma || // E019
				productTypeCode == Common.ProductType.Granulocytes || // E034
				productTypeCode == Common.ProductType.ApheresisGranulocytes || // E035
				productTypeCode == Common.ProductType.PooledGranulocytes || // E036
				productTypeCode == Common.ProductType.ApheresisGranulocytesPlatelets || // E037
				productTypeCode == Common.ProductType.Leukocytes || // E038
				productTypeCode == Common.ProductType.ApheresisLeukocytes || // E039
				productTypeCode == Common.ProductType.PooledPlasma || // E040
				productTypeCode == Common.ProductType.LiquidApheresisPlasma || // E041
				productTypeCode == Common.ProductType.PlateletRichBuffyCoat || // E042
				productTypeCode == Common.ProductType.PooledPlateletRichBuffyCoat || // E043
				productTypeCode == Common.ProductType.ApheresisLymphocytes || // E044
				productTypeCode == Common.ProductType.ApheresisMonocytes) // E045
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		#region validate unit selection TTs

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7156"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7157"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Validates if the selected unit should be allowed, should be allowed
		/// with an override or should be disallowed.
		/// Implements 3.19, TT_3.01
		/// </summary>
		/// <param name="bu"></param>
		public string ValidateWholeBloodTT301(BOL.BloodUnit bu)
		{
			string bloodUnitABORH = bu.BloodUnitMedia.AboRh.AboRHText.ToUpper();
			string messageText = string.Empty;

			switch(_patient.AboRh.Abo)
			{
				//case "O POS":
				//case "O NEG":
				case Common.ABO.O:
				{
					if(bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					return messageText;
				}
				//case "A POS":
				//case "A NEG":
				case Common.ABO.A:
				{
					if(bloodUnitABORH != "A POS" && bloodUnitABORH != "A NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override,false, messageText));
					}
					return messageText;
				}
				//case "B POS":
				//case "B NEG":
				case Common.ABO.B:
				{
					if(bloodUnitABORH != "B POS" && bloodUnitABORH != "B NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					return messageText;
				}
				//case "AB POS":
				//case "AB NEG":
				case Common.ABO.AB:
				{
					if(bloodUnitABORH != "AB POS" && bloodUnitABORH != "AB NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override,false, messageText));
					}
					return messageText;
				}
				default:
				{
					messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
					_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override,false, messageText));
					return messageText;
				}
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7099"> 
		///		<ExpectedInput>ABO/Rh compatible unit or VbecsUser with override permission</ExpectedInput>
		///		<ExpectedOutput>No warning messages or override permissable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7100"> 
		///		<ExpectedInput>ABO/Rh incompatible unit and VbecsUser with no override permission</ExpectedInput>
		///		<ExpectedOutput>Warning message and override not permissable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Validates if the selected unit should be allowed, should be allowed
		/// with an override or should be disallowed.
		/// Implements 3.19, TT_3.02
		/// </summary>
		/// <param name="bu"></param>
		public void ValidateNonWholeBloodNonPlasmaTT302(BOL.BloodUnit bu)
		{
			string bloodUnitABORH = bu.BloodUnitMedia.AboRh.AboRHText.ToUpper();
			string messageText = string.Empty;

			switch(_patient.AboRh.AboRHText.ToUpper())
			{
				case "O POS":
				case "O NEG":
				{
					if(bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "A POS":
				case "A NEG":
				{
					if(bloodUnitABORH != "A POS" &&	bloodUnitABORH != "A NEG" &&
					bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
					messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
					_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
						
					}
					break;
				}
				case "B POS":
				case "B NEG":
				{
					if(bloodUnitABORH != "B POS" && bloodUnitABORH != "B NEG" &&
						bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "AB POS":
				case "AB NEG":
				{
					if(bloodUnitABORH != "AB POS" && bloodUnitABORH != "AB NEG" && 
						bloodUnitABORH != "A POS" && bloodUnitABORH != "A NEG" &&
						bloodUnitABORH != "B POS" && bloodUnitABORH != "B NEG" &&
						bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				//ABO INCONCLUSIVE RH POS
				case "I POS":
				case "I NEG":
				{
					if(bloodUnitABORH != "O POS" && 
						bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.UnitNotCompatible().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "A INC":
				{
					
					if (bloodUnitABORH != "A POS" && bloodUnitABORH != "A NEG" &&
						bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "B INC":
				{
					if(bloodUnitABORH != "B POS" && bloodUnitABORH != "B NEG" &&
						bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "AB INC":
				{
					if(bloodUnitABORH != "AB POS" && bloodUnitABORH != "AB NEG" && 
						bloodUnitABORH != "A POS" && bloodUnitABORH != "A NEG" &&
						bloodUnitABORH != "B POS" && bloodUnitABORH != "B NEG" &&
						bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "O INC":
				{
					if(bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310, false, messageText));
					}
					break;
				}
				case "I INC":
				{
					
					if(bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				default:
				{
					if(bloodUnitABORH != "O POS" && bloodUnitABORH != "O NEG")
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7108"> 
		///		<ExpectedInput>ABO/Rh compatible unit or VbecsUser with override permission</ExpectedInput>
		///		<ExpectedOutput>No warning messages or override permissable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7109"> 
		///		<ExpectedInput>ABO/Rh incompatible and VbecsUser with no override permission</ExpectedInput>
		///		<ExpectedOutput>Warning messages and override not permissable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Validates if the selected unit should be allowed, should be allowed
		/// with an override or should be disallowed.
		/// Implements 3.19, TT_3.03A
		/// </summary>
		/// 
		/// <CR id="3097" date= "20110721" author="Jensen">
		/// Added code to include the pooled product type
		/// </CR>
		/// 
		/// <param name="bu"></param>
		public void ValidatePlasmaOnlyTT303A(BOL.BloodUnit bu)
		{
			string bloodUnitABO = bu.BloodUnitMedia.AboRh.Abo.ToString().ToUpper();
			string messageText = string.Empty;

			switch(_patient.AboRh.Abo.ToString().ToUpper())
			{
				case "O":
				{
					if(bloodUnitABO != "O" &&
						bloodUnitABO != "A" && 
						bloodUnitABO != "B" &&
						bloodUnitABO != "AB" &&
						bloodUnitABO != "MX") // CR3097
					{
						messageText = Common.StrRes.ValidMsg.UC003.UnitNotCompatible().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "A":
				{
					if(bloodUnitABO != "A" &&
						bloodUnitABO != "AB" &&
						bloodUnitABO != "MX") // CR3097
					{
						messageText = Common.StrRes.ValidMsg.UC003.UnitNotCompatible().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "B":
				{
					if(bloodUnitABO != "B" &&
						bloodUnitABO != "AB" &&
						bloodUnitABO != "MX") // CR3097
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310Override().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				case "AB":
				{
					if(bloodUnitABO != "AB" &&
						bloodUnitABO != "MX") // CR3097
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId,Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				// Emergency Issue
				case "I":
				{
					if(bloodUnitABO != "AB" &&
						bloodUnitABO != "MX") // CR3097
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
				// Emergency Issue
				default:
				{
					if(bloodUnitABO != "AB" &&
						bloodUnitABO != "MX") // CR3097
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR310().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
					}
					break;
				}
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7110"> 
		///		<ExpectedInput>ABO/Rh compatible unit or VbecsUser with override permission</ExpectedInput>
		///		<ExpectedOutput>No warning messages or override permissable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7111"> 
		///		<ExpectedInput>ABO/Rh incompatible and VbecsUser with no override permission</ExpectedInput>
		///		<ExpectedOutput>Warning messages and override not permissable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Validates if the selected unit should be allowed, should be allowed
		/// with an override or should be disallowed.
		/// Implements 3.19, TT_3.03B
		/// </summary>
		/// <param name="bu"></param>
		public void ValidatePlasmaOnlyTT303B(BOL.BloodUnit bu)
		{
			string bloodUnitABO = bu.BloodUnitMedia.AboRh.Abo.ToString().ToUpper();
			string messageText = string.Empty;
			
			if(bloodUnitABO != "O" &&
				bloodUnitABO != "A" && 
				bloodUnitABO != "B" &&
				bloodUnitABO != "AB" &&
				bloodUnitABO != "MX") // CR3097
			{
				messageText = Common.StrRes.ValidMsg.UC003.UnitNotCompatible().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR310Override, false, messageText));
			}
		}


		#endregion


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4130"> 
		///		<ExpectedInput>Valid RBC BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Validation Messages count == 0</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="4145"> 
		///		<ExpectedInput>Invalid RBC BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Error Message</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="4148"> 
		///		<ExpectedInput>Valid FFP BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Validation Messages count == 0</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="4163"> 
		///		<ExpectedInput>Invalid FFP BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Error Message</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="4164"> 
		///		<ExpectedInput>Valid CRYO BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Validation Messages count == 0</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="4165"> 
		///		<ExpectedInput>Invalid CRYO BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Error Message</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="4166"> 
		///		<ExpectedInput>Valid PLT BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Validation Messages count == 0</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="4319"> 
		///		<ExpectedInput>Invalid PLT BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Error Message</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="4320"> 
		///		<ExpectedInput>Valid Other BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Validation Messages count == 0</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="4321"> 
		///		<ExpectedInput>Invalid Other BloodUnit</ExpectedInput>
		///		<ExpectedOutput>Error Message</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7186"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7187"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// After a unit is chosen, it is sent through this method which validates it against a bunch of
		/// business rules.  There are 4 possible results of any one validation/ business rule:
		/// 1) Unit invalid with no override (Processing is stopped at this point - return false;)
		/// 2) Unit valid with override
		/// 3) Unit valid with a warning message
		/// 4) Unit valid with no contingencies (Passes all BRs)
		/// </summary>
		/// <param name="bu"></param>
		/// <param name="dateTimeSelected"></param>
		public bool ValidateUnit(BloodUnit bu, DateTime dateTimeSelected)
		{
			// Clear collection holding messages
			_validationMessages.Clear();
			string messageText = string.Empty;
			Common.ComponentClass orderComponentClass = (Common.ComponentClass) _orderedComponent.ComponentOrdered.BloodComponentClassId;
			int nCount = 0;
			#region Not Overridable

			// Invalid status
			if (bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Discarded ||
				bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Modified ||
				bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Transferred ||
				bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Transfused ||
				bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Unknown)
			{
				messageText = Common.StrRes.ValidMsg.UC003.BadStatus(Common.Utility.GetUnitStatusTextFromEnum(bu.BloodUnitStatus.UnitStatusCode)).ResString;
				_validationMessages.Add(new BOL.PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BadStatus, false, messageText));
				return false;
			}
			// Call to 3.44: 
			else if (bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Issued )
			{
				messageText = Common.StrRes.ValidMsg.UC003.BR344NotInBloodBank().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR344NotInBloodBank, false, messageText));
				return false;
			}
				

			//Call to BR_3.33/TT_3.04 
			if (!_patient.ValidateProductType(orderComponentClass, bu))
			{
				messageText = Common.StrRes.ValidMsg.UC003.BR333InvalidProductType().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR333InvalidProductType, false, messageText));
				return false;
			}

			// If emergency issue situation, the unit must be eligible for it
			if (_isEmergencyIssue)
			{
				if (!bu.EligibleForEmergencyIssue(this.DisplayOptionalCrossmatch(bu), 
					_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy ||
					_patient.ABORhSource == Common.ABORhSource.SpecimenTestDiscrepancy || 
					_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy,  // CR3114
					_orderedComponent.SpecimenIdentified && _orderedComponent.GetSpecimen().ABORhTested))  // CR3114
				{
					messageText = Common.StrRes.ValidMsg.UC003.MustEmergencyIssue().ResString;
					_validationMessages.Add(new BOL.PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.MustEmergencyIssue, false, messageText));
					return false;
				}
			}
			
			// Call to BR_3.16 (Unit ABO/Rh confirmations must be complete)
            //CR3558 - replaced RBC/WB check with PT_1.06 indicator
			if (_isFullService) 
			{
				if (bu.ProductType.AboRhConfirmationRequired && !ValidateABORhConfirmation(bu))
				{
					return false;
				}
			}

			if (bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Assigned ||
				bu.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.CrossmatchedAssigned)
			{
				DataTable dt = BOL.OrderedUnit.GetOrderedUnit(bu.BloodUnitGuid, this.OrderedComponent.DivisionCode);
				DataRow [] drArray = dt.Select(TABLE.PatientTreatment.PatientGuid + " = '" + _patient.PatientGuid + "'");
				
				if (drArray.Length > 0)
				{
					// BR_3.19 - Unit already assigned or crossmatched to this patient
					messageText = Common.StrRes.ValidMsg.UC003.AlreadySelectedForThisPatient().ResString;
					_validationMessages.Add(new BOL.PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.AlreadySelectedForThisPatient, false, messageText));
					return false;
				}
				else
				{
					// BR_3.28 - Unit has already been added for another patient
					messageText = Common.StrRes.ValidMsg.UC003.BR328AssignedToAnotherPatient().ResString;
					this._validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR328AssignedToAnotherPatient, true, messageText));
				}
			}
				
			// Call to 3.44
			if (bu.DivisionCode.Trim() != _orderedComponent.DivisionCode.Trim())
			{
				messageText = Common.StrRes.ValidMsg.UC003.BR344NotInBloodBank().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR344NotInBloodBank, false, messageText));
			}
	
			// Calls to BR_3.13, BR_3.19
			if (this.AlreadyCrossMatched(bu.BloodUnitGuid, bu.EyeReadableUnitId)) return false;
				
			// Call to BR_3.43 			 	
			if (this.QuarantinedUnit(bu)) return false;

			// Call to BR_3.45
			if (this.IsOnPendingOutgoingShipmentList(bu)) return false;

			// Call to 3.46
			if (this.IsRestrictedForDifferentPatient(bu.EyeReadableUnitId, bu.DonationType.DonationTypeCode, bu.DirectedPatient.PatientGuid)) return false;
			
			#endregion


			#region Overridable

			// Call to BR_3.23 
			IncompatibleCrossmatchResultsExist(bu);

			if (( (Common.ComponentClass) bu.BloodComponentClassId == Common.ComponentClass.WB || 
				(Common.ComponentClass) bu.BloodComponentClassId == Common.ComponentClass.RBC) 
				&& !BOL.BloodUnit.IsAutologous(bu.DonationType.DonationTypeCode) )
			{
				// Call to BR_3.06, BR_3.15
				ValidateAntibodyAntigenRequirementsMet(bu);
			}

			// Calls to BR_3.10, BR_3.19
			switch((Common.ComponentClass) bu.BloodComponentClassId)
			{
				case Common.ComponentClass.WB:
				{
					this.ValidateWholeBloodTT301(bu);
					break;
				}
				case Common.ComponentClass.RBC:
				{
					ValidateNonWholeBloodNonPlasmaTT302(bu);
					break;
				}
				case Common.ComponentClass.FFP:
				{
					this.ValidatePlasmaOnlyTT303A(bu);
					break;
				}
				case Common.ComponentClass.CRYO:
				{
					this.ValidatePlasmaOnlyTT303B(bu);
					break;
				}
			}

			// Call to 3.08
//			PatientAntiDCheck(bu);

			// Call to BR_3.19				
			ExpiredUnit(bu, dateTimeSelected);

			// Call to 3.21
			ValidateUnitRhRequirement(bu);

			// Call to BRs 3.14 All transfusion requirements except CMV Neg and 3.32 CMV Neg TRs: 
			// This is TT_3.04 portion that deals with the TR's
			System.Data.DataTable dtTRs = _patient.TransfusionRequirements;
			System.Data.DataRow [] dtRows = dtTRs.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'CR' AND RecordStatusCode = 'A'");
		
			// A user can have the same Component Requirement entered multiple times.  These 
			// bools prevent the same error message from being displayed more than once.
			bool ir		= false;
			bool waRBC	= false;
			bool waPLT	= false;
			bool leu	= false;
			bool sc		= false;
			bool cmv	= false;
			
			nCount = dtRows.GetLength(0);
			for(int nCtr = 0; nCtr < nCount; nCtr++)
			{
				System.Data.DataRow dtRow = dtRows[nCtr];
				switch((byte) dtRow[TABLE.ComponentRequirement.ComponentRequirementId])
				{
					case (int) Common.ComponentRequirement.IRRADIATE:
					{
						if (ir == false)
						{
							ValidateTT304TRsIrradiateCells(bu);
							ir = true;
						}
						break;
					}
					case (int) Common.ComponentRequirement.WASHEDRBC:
					{
						if (waRBC == false)
						{
							ValidateTT304TRsWashedRBC(bu);
							waRBC = true;
						}
						break;
					}
					case (int) Common.ComponentRequirement.WASHEDPLT:
					{
						if (waPLT == false)
						{
							ValidateTT304TRsWashedPLT(bu);
							waPLT = true;
						}
						break;
					}
					case (int) Common.ComponentRequirement.LEUKORED:
					{
						if (leu == false)
						{
							ValidateTT304TRsLeukoReduced(bu);
							leu = true;
						}
						break;
					}
                    // CR3260 
                    case (int)Common.ComponentRequirement.CMVNEG:
                    {
                        cmv = true;
                        break;
                    }
                    case (int)Common.ComponentRequirement.SCNEG:
                    {
                        sc = true;
                        break;
                    }
                }
                // CMV and Sickle now get evaluated together per BR 3.32
                if (nCtr == (nCount - 1) &&
                    (cmv == true || sc == true))
                {
                    this.ValidateTT304TRsSCAndCMVNeg(bu, cmv, sc);
                }
                // 3260 end
			}

			// BR_3.34
			// if autologous, no worries
			if (!BOL.BloodUnit.IsAutologous(bu.DonationType.DonationTypeCode))
			{
				//if directed, check for autologous
				if (BOL.BloodUnit.IsDirected(bu.DonationType.DonationTypeCode))
				{
					if (!_selectedUnits.AllAutologousUnitsChosen(bu.BloodUnitGuid))
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR334RestrictiveUnits().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR334RestrictiveUnits, true, messageText));
					}
				}
				// if restricted, check for directed and autologous
				else if (bu.DirectedPatient != null && bu.DirectedPatient.PatientGuid != Guid.Empty)
				{
					if (!_selectedUnits.AllDirectedUnitsChosen(bu.BloodUnitGuid) ||
						!_selectedUnits.AllAutologousUnitsChosen(bu.BloodUnitGuid))
					{
						messageText = Common.StrRes.ValidMsg.UC003.BR334RestrictiveUnits().ResString;
						_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR334RestrictiveUnits, true, messageText));
					}
				}

				// if allogeneic, check for restricted, directed and autologous
				else if (!_selectedUnits.AllRestrictedUnitsChosen(bu.BloodUnitGuid) || 
					!_selectedUnits.AllDirectedUnitsChosen(bu.BloodUnitGuid) || 
					!_selectedUnits.AllAutologousUnitsChosen(bu.BloodUnitGuid))
				{
					messageText = Common.StrRes.ValidMsg.UC003.BR334RestrictiveUnits().ResString;
					_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR334RestrictiveUnits, true, messageText));
				}
			}

			#endregion
			

			#region InformationalOnly

			// Call to BR_3.11
			if (!_isEmergencyIssue && _orderedComponent.SpecimenRequiredIndicator)
			{
				if (_orderedComponent.SpecimenIdentified == false || !_orderedComponent.GetSpecimen().ABORhTested)  // CR3114
				{
					messageText = Common.StrRes.ValidMsg.UC003.BR311NoABORhResults().ResString;
					_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR311NoABORhResults, true, messageText));
				}
			}

			// Call to BR_3.39
			if(bu.BloodUnitStatus.BiohazardousWarningIndicator)
			{
				messageText = Common.StrRes.ValidMsg.UC003.BR339BiohazardousWarningIndicator().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR339BiohazardousWarningIndicator, true, messageText));
			}

			// Call to BR_3.12 (if Frozen and Emergency issue stop processing here because 
			// frozen units may not be selected)
			IsFrozenUnit(bu);

			// BR_3.52
			if ((Common.ComponentClass) bu.BloodComponentClassId == Common.ComponentClass.FFP
				&&!_orderedComponent.SpecimenRequiredIndicator
				&& _patient.JustifiedAboRH)
			{
				messageText = Common.StrRes.ValidMsg.UC003.BR352JustifiedABORh().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR352JustifiedABORh, true, messageText));
			}
			
			#endregion
			
			return (_validationMessages.Count == 0);
		}

	
		/// <summary>
		///BR_3.13 	If a unit is currently selected for crossmatch for the same patient by the same or
		///other technologist, display a warning message: Unit already selected for this patient.
		///There is no override. The screen is cleared and a new unit can be selected.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="eyeReadableUnitID"></param>
		/// <returns></returns>
		private bool AlreadyCrossMatched(Guid bloodUnitGuid, string eyeReadableUnitID)
		{
			bool alreadyCrossMatched = (DAL.OrderedUnit.CheckIsAlreadyCrossMatched(bloodUnitGuid, this._orderedComponent.DivisionCode, this._patient.PatientGuid).Rows.Count != 0);
			if(alreadyCrossMatched)
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR313AlreadySelected().ResString;
				this._validationMessages.Add(new PatientUnitSelectionMessage(eyeReadableUnitID, Common.UC03UnitSelectErrorMessages.BR313AlreadySelected, false, messageText));
			}
			return alreadyCrossMatched;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7114"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7115"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// If the unit is quarantined, generate an error message.  This is not overridable under any 
		/// condition.
		/// BR_3.19 - Quarantined
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
		public bool QuarantinedUnit(BOL.BloodUnit bu)
		{	
			if(bu.BloodUnitStatus.QuarantineIndicator)
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR343UnitQuarantined().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR343UnitQuarantined, false, messageText));
			}
			return bu.BloodUnitStatus.QuarantineIndicator;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7112"> 
		///		<ExpectedInput>Non expired unit or expired unit and VbecsUser with override permission</ExpectedInput>
		///		<ExpectedOutput>false (or true with overridable warning message)</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7113"> 
		///		<ExpectedInput>Expired unit and no permissions to override</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// If the unit is expired, generate an error message.  This is not overridable under any 
		/// condition.
		/// BR_3.19 - Expired
		/// </summary>
		/// <param name="bu"></param>
		/// <param name="dateTimeSelected"></param>
		/// <returns></returns>
		public bool ExpiredUnit(BOL.BloodUnit bu, DateTime dateTimeSelected)
		{
			bool expiredUnit = (bu.BloodUnitMedia.UnitExpirationDate <= dateTimeSelected);
			if(expiredUnit)
			{
				if (this.HasAuthorityToIssueExpiredUnits())
				{
					string messageText = Common.StrRes.ValidMsg.UC003.BR319UnitExpiredOverride().ResString;
					_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR319UnitExpiredOverride, true, messageText));
				}
				else
				{
					string messageText = Common.StrRes.ValidMsg.UC003.BR319UnitExpired().ResString;
					_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR319UnitExpired, false, messageText));
				}
			}
			return expiredUnit;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/23/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7184"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7185"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// If the unit is not in the same division, generate an error message.  This is not overridable 
		/// under any condition.
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
		// BR_3.19 - In same division
		public bool InSameDivision(BOL.BloodUnit bu)
		{
			if(bu.DivisionCode.Trim() != this.OrderedComponent.DivisionCode.Trim())
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR319UnitNotInSameDivision().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR319UnitNotInSameDivision, false, messageText));
				return false;
			}
			return true;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7160"> 
		///		<ExpectedInput>A unit that can be chosen because:
		///		1) The patient has no TRs
		///		2) The patient has TRs, but unit has not tested positive
		///		3) The pateitn has TRs, unit has tested positive, but tech has override authority</ExpectedInput>
		///		<ExpectedOutput>No messages or messages that are overridable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7161"> 
		///		<ExpectedInput>Unit that cannot be chosed because it has tested positive to antigen and
		///		tech doesn't have override authority</ExpectedInput>
		///		<ExpectedOutput>Message that is not overridable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This validates that the chosen unit meets the antigen negative requirements 
		/// for the patient.  If a unit doesn't meet patient requirements, it can still 
		/// be selected if the user has proper authority.
		/// </summary>
		/// <param name="bu"></param>
		// BR_3.06, BR_3.15
		public void ValidateAntibodyAntigenRequirementsMet(BOL.BloodUnit bu)
		{
			string messageText = string.Empty;
			ArrayList bloodUnitAntigens = _patient.ValidateAntibodyAntigenRequirementsMet(bu);

			foreach (BOL.AntigenNegativeMessage aNM in bloodUnitAntigens)
			{
				if (!aNM.Tested)
				{
					messageText = Common.StrRes.ValidMsg.UC003.BR315UnitNotTyped(aNM.AntigenName).ResString;
					_validationMessages.Add( new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR315UnitNotTyped, true, messageText) );
				}
				else
				{
					if (this.HasAuthorityToIssueAntigenPositiveOrUntypedRBCs())
					{
						// Antigen
						if (aNM.AntibodyName == string.Empty)
						{
							messageText = Common.StrRes.ValidMsg.UC003.BR306AntigenOverride(aNM.AntigenName, bu.EyeReadableUnitId + ", " + bu.ProductName).ResString;
							_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR306Override, true, messageText));
						}
						else // antibody
						{
							messageText = Common.StrRes.ValidMsg.UC003.BR306AntibodyOverride(aNM.AntibodyName, bu.EyeReadableUnitId + ", " + bu.ProductName).ResString;
							_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR306Override, true, messageText));
						}
					}
					else
					{
						// Antigen
						if (aNM.AntibodyName == string.Empty)
						{
							_validationMessages.Clear();
							messageText = Common.StrRes.ValidMsg.UC003.BR306Antigen(aNM.AntigenName, bu.EyeReadableUnitId + ", " + bu.ProductName).ResString;
							_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR306, false, messageText));
						}
						else // antibody
						{
							_validationMessages.Clear();
							messageText = Common.StrRes.ValidMsg.UC003.BR306Antibody(aNM.AntibodyName, bu.EyeReadableUnitId + ", " + bu.ProductName).ResString;
							_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR306, false, messageText));
						}
					}
				}
			}
			bloodUnitAntigens = null;
		}


		


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7168"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7169"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This returns a boolean value indicating if the selected unit is frozen.
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
		// BR_3.12
		public bool IsFrozenUnit(BOL.BloodUnit bu)
		{
			if (bu.IsFrozen)
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR312UnitFrozen().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR312UnitFrozen, true, messageText));
				return true;
			}
			else
			{
				return false;
			}
		}
       
		/// <summary>
		/// This returns a boolean value indicating if the selected unit has had ABO/Rh tests confirmed.
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
		// BR_3.16
		private bool ValidateABORhConfirmation(BloodUnit bu)
		{
			if (bu.BloodUnitMedia.DoAboRHConfirmationsMatch(bu.DivisionCode))
			{
				return true;
			}
			else
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR316ABORhConfirmation().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR316ABORhConfirmation, false, messageText));
				return false;
			}
		}
        
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7158"> 
		///		<ExpectedInput>An Rh Positve patient and an Rh Negative Unit</ExpectedInput>
		///		<ExpectedOutput>No messages generated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7159"> 
		///		<ExpectedInput>An Rh Negatve patient and an Rh Positive Unit</ExpectedInput>
		///		<ExpectedOutput>An overridable message generated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This returns a boolean value indicating if the selected unit has met Rh
		/// compatibility requirements.
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
        // CR3261
        public void ValidateUnitRhRequirement(BOL.BloodUnit bu)
        {
            Common.ProductType productTypeCode = Common.Utility.GetProductTypeFromProductTypeCode(bu.ProductType.ProductTypeCode);

            if (!_patient.IsUnitRhSatisfactory(bu))
            {
                string messageText = Common.StrRes.ValidMsg.UC003.BR321().ResString;
                _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR321, true, messageText));
            }
        }

		/// <summary>
		/// This returns a boolean value indicating if the selected unit has had 
		/// incompatible crossmatch results in the past.
		/// </summary>
		/// <param name="bu"></param>
		/// <returns></returns>
		// BR_3.23
		private bool IncompatibleCrossmatchResultsExist(BOL.BloodUnit bu)
		{
			if (BOL.OrderedUnit.IncompatibleCrossmatchResultsExist(_patient.PatientGuid, bu.BloodUnitGuid))
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR323UnitIncompatible().ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR323UnitIncompatible, true, messageText));
				return true;
			}
			return false;
		}


	


        ///<Developers>
        ///	<Developer>C. Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/22/2005</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="7162"> 
        ///		<ExpectedInput>Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///<Case type="1" testid ="7163"> 
        ///		<ExpectedInput>Invalid Parameters</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// This is the part of the implementation of TT_3.04 that deals with Sickle Cell
        /// </summary>
        /// <param name="bu"></param>
        /// <param name="cmv"></param>
        /// <param name="sc"></param>
        /// BR_3.14, TT_3.04
        /// CR3260: Sickle cell now requires an override.
        public void ValidateTT304TRsSCAndCMVNeg(BOL.BloodUnit bu, bool cmv, bool sc)
        {
            string messageText = string.Empty;

            bool cmvStatusSatisfied = false;
            bool sickleCellStatusSatisfied = false;

            cmvStatusSatisfied = (cmv == false || bu.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG));
            sickleCellStatusSatisfied = (sc == false || bu.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG));

            // Both requirements unsatisfied
            if (!sickleCellStatusSatisfied && !cmvStatusSatisfied)
            {
                if (this.HasAuthorityToIssueAntigenPositiveOrUntypedRBCs())
                {
                    messageText = Common.StrRes.ValidMsg.UC003.BR332SickleCellAndCMVNegRequiredOverride().ResString;
                    _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR332SickleCellAndCMVNegRequiredOverride, true, messageText));
                }
                else
                {
                    messageText = Common.StrRes.ValidMsg.UC003.BR332SickleCellAndCMVNegRequired().ResString;
                    _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR332SickleCellAndCMVNegRequired, false, messageText));
                }
            }
            // Sickle cell only
            else if (!sickleCellStatusSatisfied)
            {
                if (this.HasAuthorityToIssueAntigenPositiveOrUntypedRBCs())
                {
                    messageText = Common.StrRes.ValidMsg.UC003.BR332SickleCellNegRequiredOverride().ResString;
                    _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR332SickleCellNegRequiredOverride, true, messageText));
                }
                else
                {
                    messageText = Common.StrRes.ValidMsg.UC003.BR332SickleCellNegRequired().ResString;
                    _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR332SickleCellNegRequired, false, messageText));
                }
            }
            // CMV only
            else if (!cmvStatusSatisfied)
            {
                if (this.HasAuthorityToIssueAntigenPositiveOrUntypedRBCs())
                {
                    messageText = Common.StrRes.ValidMsg.UC003.BR332CMVNegRequiredOverride().ResString;
                    _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR332CMVNegRequiredOverride, true, messageText));
                }
                else
                {
                    messageText = Common.StrRes.ValidMsg.UC003.BR332CMVNegRequired().ResString;
                    _validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR332CMVNegRequired, false, messageText));
                }
            }
        }




		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7148"> 
		///		<ExpectedInput>Unit that will satisfy the patient's leuko requirement</ExpectedInput>
		///		<ExpectedOutput>No messages generated</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7149"> 
		///		<ExpectedInput>Unit that does not satisfy the patient's leuko requirement</ExpectedInput>
		///		<ExpectedOutput>Message generated</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the part of the implementation of TT_3.04 that deals with Leukocyte Reduction.
		/// </summary>
		/// <param name="bu"></param>
		/// BR_3.14, TT_3.04
		public void ValidateTT304TRsLeukoReduced(BOL.BloodUnit bu)
		{			
			if (!bu.IsComponentRequirementSatisfied(Common.ComponentRequirement.LEUKORED))
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR314TransfusionRequirements(": Leuko Reduced").ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR314TransfusionRequirements, true, messageText));
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7150"> 
		///		<ExpectedInput>Unit that will satisfy the patient's washed platelet requirement</ExpectedInput>
		///		<ExpectedOutput>No messages generated</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7151"> 
		///		<ExpectedInput>Unit that does not satisfy the patient's washed platelet requirement</ExpectedInput>
		///		<ExpectedOutput>Message generated</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the part of the implementation of TT_3.04 that deals with Washed Platelets.
		/// </summary>
		/// <param name="bu"></param>
		/// BR_3.14, TT_3.04
		public void ValidateTT304TRsWashedPLT(BOL.BloodUnit bu)
		{
			if (!bu.IsComponentRequirementSatisfied(Common.ComponentRequirement.WASHEDPLT))
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR314TransfusionRequirements(": Washed").ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR314TransfusionRequirements, true, messageText));
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7152"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7153"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the part of the implementation of TT_3.04 that deals with Leukocyte Reduction.
		/// </summary>
		/// <param name="bu"></param>
		/// BR_3.14, TT_3.04
		public void ValidateTT304TRsWashedRBC(BOL.BloodUnit bu)
		{
			if (!bu.IsComponentRequirementSatisfied(Common.ComponentRequirement.WASHEDRBC))
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR314TransfusionRequirements(": Washed").ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR314TransfusionRequirements, true, messageText));
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7154"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7155"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the part of the implementation of TT_3.04 that deals with Irradiated Cells.
		/// BR_3.14, TT_3.04
		/// </summary>
		/// <param name="bu"></param>
		public void ValidateTT304TRsIrradiateCells(BOL.BloodUnit bu)
		{
			if (!bu.IsComponentRequirementSatisfied(Common.ComponentRequirement.IRRADIATE))
			{
				string messageText = Common.StrRes.ValidMsg.UC003.BR314TransfusionRequirements(": Irradiated").ResString;
				_validationMessages.Add(new PatientUnitSelectionMessage(bu.EyeReadableUnitId, Common.UC03UnitSelectErrorMessages.BR314TransfusionRequirements, true, messageText));
			}
		}


		/// <summary>
		/// Removes units added in a previous session from the picklist.
		/// </summary>
		/// <param name="dtInitialUnits"></param>
		/// <returns></returns>
		private DataTable RemoveUnitsAlreadyAdded(DataTable dtInitialUnits)
		{
			// Get the units that were added in a previous session
			DataTable dtAlreadyOrderedUnits = BOL.OrderedUnit.GetOrderedUnitByComponent(this._orderedComponent.OrderedComponentGuid, this._orderedComponent.DivisionCode);
			ArrayList unitsForRemoval = new ArrayList();
		
			foreach (DataRow drAlreadyOrderedUnit in dtAlreadyOrderedUnits.Rows)
			{
				DataRow [] drAlreadySelected = dtInitialUnits.Select("BloodUnitGuid = '" + ((Guid)drAlreadyOrderedUnit["BloodUnitGuid"]) + "'");
				if (drAlreadySelected.Length > 0) dtInitialUnits.Rows.Remove(drAlreadySelected[0]);
			}
			return dtInitialUnits;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7166"> 
		///		<ExpectedInput>Nothing</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8397"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			throw new BOL.BusinessObjectException("This is not implemented and should not be called!");
		}

		
		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/22/2005</CreationDate>
		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			//This class doesn't really have a need for this method
			throw new BOL.BusinessObjectException("This is not implemented and should not be called!");
		}



		#endregion
        
		
		#region selected units inner class
		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/20/2003</CreationDate>
		///
		/// <summary>
		/// This inner class manages the units (as well as corresponding comments) selected for
		/// the OrderedComponent
		/// </summary>
		public class SelectedUnits : ICollection
		{
			#region variables

			private int _alreadySelectedUnitsCrossmatchedSuccessfully;
			//
			private Hashtable _bloodUnitsSelected;
			//
			private PatientUnitSelection _patientUnitSelection;
			//
			private DataTable _dtOrderedUnits;
			//
			private DataTable _dtTempOrderedUnitComments;
			//
			private DataTable _dtOrderedUnitComments;
			//
			private Hashtable _lockedUnits;
			//
			private DataTable _dtDirectedBloodUnits;
			//
			private ArrayList _directedUnitsAvailableForSelection;
			private ArrayList _autologousUnitsAvailableForSelection;
			private ArrayList _restrictedUnitsAvailableForSelection;
			//
			private DataTable _dtAlreadySelectedUnits;
            
			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>9/20/2003</CreationDate>
			///
			///<summary>
			///When a unit is added that generates warning messages, this event
			///is raised.
			///</summary>
			///public event EventHandler ValidationMessagesGenerated;
			/// <summary>
			/// Issues messages
			/// </summary>
			public delegate void SelectMessagesEventHandler(object sender, SelectEventArgs e);

			/// <summary>
			/// Event that is raised when messages are generated
			/// </summary>
			public event SelectMessagesEventHandler SelectMessagesEvent;

			
			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>9/20/2003</CreationDate>
			///
			///<summary>
			///When a unit is added that generates warning messages, this event
			///is raised.
			///</summary>
			///<param name="bloodUnitGuid"></param>
			///<param name="bloodUnitStatusGuid"></param>
			protected virtual void SelectMessagesGenerated(Guid bloodUnitGuid, Guid bloodUnitStatusGuid)
			{
				SelectEventArgs sEA = new SelectEventArgs(bloodUnitGuid, bloodUnitStatusGuid);
				if (SelectMessagesEvent != null)
					SelectMessagesEvent(this, sEA);
			}

			#endregion


			#region constructors

			///<Developers>
			///	<Developer>COE User</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/20/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3484"> 
			///		<ExpectedInput>Valid instance of PatientUnitSelection</ExpectedInput>
			///		<ExpectedOutput>Valid SelectedUnits object</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3485"> 
			///		<ExpectedInput>null instance of PatientUnitSelection</ExpectedInput>
			///		<ExpectedOutput>ArgumentException</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			/// This is the main constructor.
			///</summary>
			public SelectedUnits(PatientUnitSelection patientUnitSelection)
			{
				
				_patientUnitSelection = patientUnitSelection;
				//
				_bloodUnitsSelected = new Hashtable();
				//
				_dtOrderedUnits = BOL.OrderedUnit.InitializeOrderedUnitTable();
				//
				_dtTempOrderedUnitComments = BOL.OrderedUnitComment.InitializeBlankOrderedUnitCommentTable();
				//
				_dtOrderedUnitComments = BOL.OrderedUnitComment.InitializeBlankOrderedUnitCommentTable();
				// Keeps track of BloodUnitStatusGUIDs in order to release locks
				_lockedUnits = new Hashtable();
				// BR_3.31
				_dtAlreadySelectedUnits = BOL.OrderedUnit.GetOrderedUnitByComponent(_patientUnitSelection._orderedComponent.OrderedComponentGuid, _patientUnitSelection._orderedComponent.DivisionCode);


				// BR_3.25 - Get units directed to the patient
				_dtDirectedBloodUnits = BOL.BloodUnitMedia.GetBloodUnitsByPatient(_patientUnitSelection._patient.PatientGuid);

				// only want available units:
				ArrayList rowsForRemoval = new ArrayList();
				for (int i=0; i<_dtDirectedBloodUnits.Rows.Count; i++)
				{
					int unitComponentClass = (int) _dtDirectedBloodUnits.Rows[i][TABLE.ComponentClass.ComponentClassId];

					if (_patientUnitSelection._orderedComponent.ComponentOrdered.BloodComponentClassId == (int) Common.ComponentClass.RBC ||
						_patientUnitSelection._orderedComponent.ComponentOrdered.BloodComponentClassId == (int) Common.ComponentClass.WB)
					{
						if (unitComponentClass != (int) Common.ComponentClass.WB && unitComponentClass != (int) Common.ComponentClass.RBC)
						{
							rowsForRemoval.Add(_dtDirectedBloodUnits.Rows[i]);
						}
					}
					else if ((int) _patientUnitSelection._orderedComponent.ComponentOrdered.BloodComponentClassId != unitComponentClass)
					{
						rowsForRemoval.Add(_dtDirectedBloodUnits.Rows[i]);
					}
				}
						
				
				for (int i=0; i<_dtDirectedBloodUnits.Rows.Count; i++)
				{
					if ( ((string) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode]).Trim() != "A" &&
						((string) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode]).Trim() != "L")
					{
						if (!rowsForRemoval.Contains(_dtDirectedBloodUnits.Rows[i])) rowsForRemoval.Add(_dtDirectedBloodUnits.Rows[i]);
					}
				}
				for (int i=0; i<rowsForRemoval.Count; i++)
				{
					_dtDirectedBloodUnits.Rows.Remove( (DataRow) rowsForRemoval[i]);
				}
				// BR_3.34
				_directedUnitsAvailableForSelection = new ArrayList();
				_autologousUnitsAvailableForSelection = new ArrayList();
				_restrictedUnitsAvailableForSelection = new ArrayList();
				for (int i=0; i<_dtDirectedBloodUnits.Rows.Count; i++)
				{
					if (_dtDirectedBloodUnits.Rows[i][TABLE.BloodUnit.DivisionCode].ToString().Trim() == Common.LogonUser.LogonUserDivisionCode.Trim())
					{
						if (Common.Utility.IsAutologous( (int) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnit.DonationTypeId]))
						{
							_autologousUnitsAvailableForSelection.Add((Guid) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnit.BloodUnitGuid]);
						}
						else if (Common.Utility.IsDirected((int) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnit.DonationTypeId]))
						{
							_directedUnitsAvailableForSelection.Add( (Guid) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnit.BloodUnitGuid]);
						}
						else
						{
							_restrictedUnitsAvailableForSelection.Add( (Guid) _dtDirectedBloodUnits.Rows[i][TABLE.BloodUnit.BloodUnitGuid]);
						}
					}
				}
				//
				RemoveAlreadySelectedUnits();
			}

			#endregion


			#region EventArgs class
		
			/// <summary>
			/// This event holds messages related to the issue of units.
			/// Lohse 02/09/05 - Adding BloodUnitStatusGuid - for locking needs
			/// </summary>
			public class SelectEventArgs : System.EventArgs 
			{

				private Guid _bloodUnitGuid;
				private Guid _bloodUnitStatusGuid;

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/14/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7611"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7612"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// Constructor
				/// </summary>
				/// <param name="bloodUnitGuid"></param>
				/// <param name="bloodUnitStatusGuid"></param>
				public SelectEventArgs(Guid bloodUnitGuid, Guid bloodUnitStatusGuid) 
				{
					_bloodUnitGuid = bloodUnitGuid;
					_bloodUnitStatusGuid = bloodUnitStatusGuid;
				}

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/14/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7613"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7614"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// BloodUnitStatusGuid
				/// </summary>
				public Guid BloodUnitStatusGuid
				{
					get
					{
						return _bloodUnitStatusGuid;
					}
				}

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/14/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7615"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7616"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// BloodUnitGuid
				/// </summary>
				public Guid BloodUnitGuid
				{
					get
					{
						return _bloodUnitGuid;
					}
				}
			}
		
			#endregion


			#region properties


			///<Developers>
			///	<Developer>Carrie Van Stedum</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/27/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6602"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6603"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// AlreadySelectedUnits
			/// </summary>
			public DataTable AlreadySelectedUnits
			{
				get
				{
					return _dtAlreadySelectedUnits;
				}
			}

			///<Developers>
			///	<Developer>Carrie Van Stedum</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/27/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6604"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6605"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// DirectedUnits
			/// </summary>
			public DataTable DirectedUnits
			{
				get
				{
					return _dtDirectedBloodUnits;
				}
			}

			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/20/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3486"> 
			///		<ExpectedInput>N/A</ExpectedInput>
			///		<ExpectedOutput>N/A</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3487"> 
			///		<ExpectedInput>N/A</ExpectedInput>
			///		<ExpectedOutput>N/A</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This is used to store OrderedUnitComments generated by the UI.  These comments are 
			/// added to _dtOrderedUnitComments for saving later on.
			/// </summary>
			public DataTable TempOrderedUnitComments
			{
				set
				{
					_dtTempOrderedUnitComments = value;
				}
				get
				{
					return _dtTempOrderedUnitComments;
				}
			}

			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/20/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3488"> 
			///		<ExpectedInput>None</ExpectedInput>
			///		<ExpectedOutput>1</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3489"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			///This method keeps track of the number of units that have been crossmatched.
			///This is used to determine whether to launch the Crossmatch functionality at 
			///the end of processing.
			///</summary>
			///
			public int AlreadySelectedUnitsCrossmatchedSuccessfully
			{
				set
				{
					_alreadySelectedUnitsCrossmatchedSuccessfully = value;
				}
				get
				{
					return _alreadySelectedUnitsCrossmatchedSuccessfully;
				}
			}

			#endregion


			#region methods

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/14/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7617"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7618"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// AllDirectedUnitsChosen
			/// </summary>
			/// <returns></returns>
			public bool AllDirectedUnitsChosen(Guid bloodUnitGuid)
			{
				int count = 0;
				
				for (int i=0; i<_directedUnitsAvailableForSelection.Count; i++)
				{
					if (this.Contains( (Guid) _directedUnitsAvailableForSelection[i]) ||
						bloodUnitGuid == (Guid) _directedUnitsAvailableForSelection[i])
					{
						count++;
					}
				}
				return count == _directedUnitsAvailableForSelection.Count;
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/14/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7619"> 
			///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
			///		<ExpectedOutput>boolean</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7620"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// AllAutologousUnitsChosen
			/// </summary>
			/// <returns></returns>
			public bool AllAutologousUnitsChosen(Guid bloodUnitGuid)
			{
				int count = 0;
							
				for (int i=0; i<_autologousUnitsAvailableForSelection.Count; i++)
				{
					if (this.Contains( (Guid) _autologousUnitsAvailableForSelection[i]) ||
						bloodUnitGuid == (Guid) _autologousUnitsAvailableForSelection[i])
					{
						count++;
					}
				}
				return count == _autologousUnitsAvailableForSelection.Count;
			}


			///<Developers>
			///	<Developer>COE User</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/14/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="8345"> 
			///		<ExpectedInput>BloodUnit Guid</ExpectedInput>
			///		<ExpectedOutput>boolean</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="8346"> 
			///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
			///		<ExpectedOutput>false</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// AllRestrictedUnitsChosen
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			/// <returns></returns>
			public bool AllRestrictedUnitsChosen(Guid bloodUnitGuid)
			{
				int count = 0;
				
				for (int i=0; i<_restrictedUnitsAvailableForSelection.Count; i++)
				{
					if (this.Contains( (Guid) _restrictedUnitsAvailableForSelection[i]) ||
						bloodUnitGuid == (Guid) _restrictedUnitsAvailableForSelection[i])
					{
						count++;
					}
				}
				return count == _restrictedUnitsAvailableForSelection.Count;
			}

			/// <summary>
			/// Removes units that have already been selected for the patient so they are not
			/// forced to pick them again.
			/// </summary>
			private void RemoveAlreadySelectedUnits()
			{
				foreach (DataRow dtRow in _dtAlreadySelectedUnits.Rows)
				{
					DataRow [] dtRowArray = _dtDirectedBloodUnits.Select(TABLE.BloodUnit.BloodUnitGuid + " = '" + dtRow[TABLE.BloodUnit.BloodUnitGuid] + "'" );
					if (dtRowArray.Length > 1) 
					{
						throw new BOL.BusinessObjectException("A unit is selected twice for a patient in this order; not right!");
					}
					else if (dtRowArray.Length == 1)
					{
						_dtDirectedBloodUnits.Rows.Remove(dtRowArray[0]);
					}
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>2/15/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6661"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6662"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Contains
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			/// <returns></returns>
			public bool Contains(Guid bloodUnitGuid)
			{
				if (_bloodUnitsSelected.Contains(bloodUnitGuid))
				{
					return true;
				}
				else
				{
					return false;
				}
			}

			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>9/29/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3470"> 
			///		<ExpectedInput>Valid and not null objects</ExpectedInput>
			///		<ExpectedOutput>Unit is added successfully</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3471"> 
			///		<ExpectedInput>1 or more objects are null</ExpectedInput>
			///		<ExpectedOutput>ArgumentException</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			///This method is used to add methods to the SelectedUnits collection.
			///</summary>
			public bool AddUnit(BloodUnit bloodUnit, System.DateTime dateTimeSelected, string assigningTech, Common.UserRoleType userRole)
			{
				if (bloodUnit == null)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnit").ResString);
				}
				if (dateTimeSelected == DateTime.MinValue)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("dateTimeSelected").ResString);
				}
				if (userRole == Common.UserRoleType.Unknown)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("userRole").ResString);
				}

				_dtTempOrderedUnitComments.Clear();
				
				if (!_patientUnitSelection.ValidateUnit(bloodUnit, dateTimeSelected))	
				{
					SelectMessagesGenerated(bloodUnit.BloodUnitGuid, bloodUnit.BloodUnitStatus.BloodUnitStatusGuid);
				}
				
				// This will be 0 if the unit was fine or the user has overridden all error messages.
				if (_patientUnitSelection.ValidationMessages.Count == 0)
				{
					_bloodUnitsSelected.Add(bloodUnit.BloodUnitGuid, bloodUnit);
					_lockedUnits.Add(bloodUnit.BloodUnitGuid,bloodUnit.BloodUnitStatus.BloodUnitGuid);
					//
					AddOrderedUnit(bloodUnit, dateTimeSelected, assigningTech);
					//
//					OnDirectedUnitAddedOrRemoved();
					return true;
				}
				else
				{
					//					BOL.BOLUtility.UnlockRecord(bloodUnit.BloodUnitStatus.BloodUnitStatusGuid, Common.LockFunctions.UC003_SelectUnitsForPatient);
					_lockedUnits.Remove(bloodUnit.BloodUnitGuid);
					return false;
				}
			}
			
			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>9/29/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3472"> 
			///		<ExpectedInput>Valid bloodUnitGuid</ExpectedInput>
			///		<ExpectedOutput>Unit is removed from collection</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3473"> 
			///		<ExpectedInput>Empty Guid</ExpectedInput>
			///		<ExpectedOutput>ArgumentException</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			///This is called when the user has deselected a unit.
			///</summary>
			public void RemoveUnit(Guid bloodUnitGuid)
			{				
				if (bloodUnitGuid == Guid.Empty)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
				}
				
				_bloodUnitsSelected.Remove(bloodUnitGuid);
				_patientUnitSelection.RemoveOptionalCrossmatch(bloodUnitGuid);
				
				// Release the lock
				if (_lockedUnits.Contains(bloodUnitGuid))
				{
					Guid bloodUnitStatusGUID = (Guid) this._lockedUnits[bloodUnitGuid];
					//						BOL.BOLUtility.UnlockRecord(bloodUnitStatusGUID, Common.LockFunctions.UC003SelectUnitsForPatient);
					_lockedUnits.Remove(bloodUnitGuid);
				}
		
				this.RemoveOrderedUnit(bloodUnitGuid);
			}
			
			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>9/29/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3474"> 
			///		<ExpectedInput>Valid bloodUnitGuid</ExpectedInput>
			///		<ExpectedOutput>valid BloodUnit</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3475"> 
			///		<ExpectedInput>Empty Guid</ExpectedInput>
			///		<ExpectedOutput>ArgumentException</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			/// This retrieves a blood unit from the collection.
			///</summary>
			public BOL.BloodUnit GetUnit(Guid bloodUnitGUID)
			{
				if (bloodUnitGUID == Guid.Empty)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("OrderedComponent, VbecsUser").ResString);
				}
	
				IDictionaryEnumerator enumerator = _bloodUnitsSelected.GetEnumerator();
				BOL.BloodUnit bloodUnit = null;
				while ( enumerator.MoveNext() ) 
				{	
					if (enumerator.Key.Equals(bloodUnitGUID))
					{
						bloodUnit = (BOL.BloodUnit) enumerator.Value;
						break;
					}
				}
				return bloodUnit;
			}
			
			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>9/29/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3476"> 
			///		<ExpectedInput>Valid instance of bloodUnitGuid</ExpectedInput>
			///		<ExpectedOutput>DataRow</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3477"> 
			///		<ExpectedInput>empty Guid</ExpectedInput>
			///		<ExpectedOutput>ArgumentException</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// GetOrderedUnit
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			/// <returns></returns>
			public DataRow GetOrderedUnit(Guid bloodUnitGuid)
			{
				if (bloodUnitGuid == Guid.Empty)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
				}

				DataRow	[] dr = _dtOrderedUnits.Select(TABLE.BloodUnit.BloodUnitGuid + " = '" + bloodUnitGuid + "'");
				if (dr.Length==1)
				{
					return dr[0];
				}
				else if(dr.Length==0)
				{
					// try to get from previous session
					dr = this._dtAlreadySelectedUnits.Select(TABLE.BloodUnit.BloodUnitGuid + " = '" + bloodUnitGuid + "'");
					if (dr.Length==1)
					{
						return dr[0];
					}
					else
					{
						return null;
					}
				}
				else
				{
					throw new BOL.BusinessObjectException("More than one row returned!");
				}
			}

			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/20/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3490"> 
			///		<ExpectedInput>Valid arguments</ExpectedInput>
			///		<ExpectedOutput>true</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3491"> 
			///		<ExpectedInput>Invalid arguments</ExpectedInput>
			///		<ExpectedOutput>false</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///<param name="lastUpdateFunctionId">UC or calling method</param>
			///<param name="bloodUnitSelectionExceptions">exceptions generated by blood unit selection</param>
			///<param name="workloadProcessID">Workload ProcessID</param>
			///<param name="dtBCEMessages">BCE Messages</param>
			/// <summary>
			/// Saves the units for one orderedComponent.  Note that one patient may have > 1 Ordered Components.
			/// </summary>
			public bool SaveUnits(Common.UpdateFunction lastUpdateFunctionId, ArrayList bloodUnitSelectionExceptions, ref Common.WorkloadProcessID workloadProcessID, DataTable dtBCEMessages)
			{
				//CR 2941 Added dtBCEMessges
				if (_dtOrderedUnits.Rows.Count > 0)
				{
					// Only need to update OrderedComponent when a specimen has been identified...
					DataTable dtOrderedComponents = BOL.OrderedComponent.InitializeOrderedComponentTable();
					if (_patientUnitSelection.SpecimenIdentified || _patientUnitSelection.OptionalCrossmatchOrdered())
					{
						DataRow drCs = dtOrderedComponents.NewRow();
						drCs[TABLE.OrderedComponent.OrderedComponentGuid]	= _patientUnitSelection.OrderedComponent.OrderedComponentGuid;
											
						drCs[TABLE.OrderedComponent.PatientSpecimenGuid]	= _patientUnitSelection.OrderedComponent.GetSpecimen().SpecimenGuid;  // CR3114
						drCs[TABLE.OrderedComponent.SpecimenWorkloadUid]	= _patientUnitSelection.OrderedComponent.GetSpecimen().SpecimenUid;  // CR3114
						
						drCs[TABLE.OrderedComponent.OrderStatusCode]		= _patientUnitSelection.OrderedComponent.OrderStatusCode;
						drCs[TABLE.OrderedComponent.SpecimenAccessionNumber]	= _patientUnitSelection.OrderedComponent.SpecimenAccessionNumber;
						drCs[TABLE.OrderedComponent.NoInappropriateReportText] = _patientUnitSelection.OrderedComponent.NoInappropriateReportText;
						drCs[TABLE.OrderedComponent.InappropriateRequestIndicator] = _patientUnitSelection.OrderedComponent.InappropriateRequestIndicator;
						drCs[TABLE.OrderedComponent.EmergencyOrderIndicator]= false;
						drCs[TABLE.OrderedComponent.LastUpdateUser]		= Common.LogonUser.LogonUserName;
						drCs[TABLE.OrderedComponent.RowVersion]			= _patientUnitSelection.OrderedComponent.RowVersion;
						dtOrderedComponents.Rows.Add(drCs);
					}
						
					DataTable dtBloodUnitSelectionExceptions = null;

					foreach(BOL.ExceptionReport exRep in bloodUnitSelectionExceptions)
					{
						if (dtBloodUnitSelectionExceptions == null)
							dtBloodUnitSelectionExceptions = exRep.ExceptionData.Table.Clone();
						dtBloodUnitSelectionExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
					}
					System.Data.DataTable dtWorkloadEvents = null;
					System.Data.DataTable dtWorkloadTemp = null;
					System.Guid [] patientGuids = new Guid[1];
					patientGuids[0] = this._patientUnitSelection.OrderedComponent.Patient.PatientGuid;
					foreach(System.Data.DataRow dtRow in _dtOrderedUnits.Rows)
					{
						if (!dtRow.IsNull(Common.VbecsTables.OrderedUnit.CrossmatchResultCode))
						{
							if(dtRow[Common.VbecsTables.OrderedUnit.CrossmatchResultCode].ToString() == Common.Utility.GetCrossmatchResultStringFromEnum(Common.CrossmatchResult.CompatibleEXM))
							{
								if (dtWorkloadEvents == null)
								{
									dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.CrossmatchUnitElectronic,  patientGuids, Common.WorkloadTransactionType.Patient, false);
								}
								else
								{
									dtWorkloadTemp = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.CrossmatchUnitElectronic, patientGuids, Common.WorkloadTransactionType.Patient, false);
									if (dtWorkloadTemp.Rows.Count > 0)
									{
										dtWorkloadEvents.Rows.Add(dtWorkloadTemp.Rows[0].ItemArray);
									}
								}

							}
						}
						workloadProcessID = Common.WorkloadProcessID.CrossmatchUnitElectronic;
					}
					dtWorkloadEvents = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
					return DAL.PatientUnitSelection.SaveUnits(_dtOrderedUnits, _dtOrderedUnitComments, dtOrderedComponents, lastUpdateFunctionId, dtBloodUnitSelectionExceptions, dtWorkloadEvents, dtBCEMessages);
				}
				else if (_patientUnitSelection.SpecimenIdentified)
				{
					DataTable dtOrderedComponents = BOL.OrderedComponent.InitializeOrderedComponentTable();
					DataRow drCs = dtOrderedComponents.NewRow();
					
					drCs[TABLE.OrderedComponent.OrderedComponentGuid]	= _patientUnitSelection.OrderedComponent.OrderedComponentGuid;

					drCs[TABLE.OrderedComponent.PatientSpecimenGuid]	= _patientUnitSelection.OrderedComponent.GetSpecimen().SpecimenGuid;  // CR3114
					drCs[TABLE.OrderedComponent.SpecimenWorkloadUid]	= _patientUnitSelection.OrderedComponent.GetSpecimen().SpecimenUid;  // CR3114
					
					drCs[TABLE.OrderedComponent.OrderStatusCode]		= _patientUnitSelection.OrderedComponent.OrderStatusCode;
					drCs[TABLE.OrderedComponent.SpecimenAccessionNumber]	= _patientUnitSelection.OrderedComponent.SpecimenAccessionNumber;
					drCs[TABLE.OrderedComponent.NoInappropriateReportText] = _patientUnitSelection.OrderedComponent.NoInappropriateReportText;
					drCs[TABLE.OrderedComponent.InappropriateRequestIndicator] = _patientUnitSelection.OrderedComponent.InappropriateRequestIndicator;
					drCs[TABLE.OrderedComponent.LastUpdateUser]		= Common.LogonUser.LogonUserName;
					drCs[TABLE.OrderedComponent.RowVersion]			= _patientUnitSelection.OrderedComponent.RowVersion;
					dtOrderedComponents.Rows.Add(drCs);
					Common.Utility.AppendLastUpdateInformation(dtOrderedComponents, lastUpdateFunctionId);
					return DAL.OrderedComponent.UpdateStatus(dtOrderedComponents, null, null);
				}
				else
				{
					return true;
				}
			}

			/// <summary>
			/// When a unit is added to SelectedUnits, a corresponding OrderedUnit is also 
			/// added.  The OrderedUnit is what actually gets saved to the DB and is the whole
			/// point of this UC.
			/// </summary>
			/// <param name="bloodUnit"></param>
			/// <param name="dateTimeSelected"></param>
			/// <param name="assigningTech"></param>
			private void AddOrderedUnit(BOL.BloodUnit bloodUnit, System.DateTime dateTimeSelected, string assigningTech)
			{
				DataRow dr = _dtOrderedUnits.NewRow();
				
				dr[TABLE.OrderedUnit.OrderedUnitGuid]		= Guid.NewGuid();
				dr[TABLE.OrderedUnit.OrderedComponentGuid]	= _patientUnitSelection._orderedComponent.OrderedComponentGuid;
				dr[TABLE.OrderedUnit.BloodUnitGuid]			= bloodUnit.BloodUnitGuid;
				dr[TABLE.OrderedUnit.SelectedDate]			= dateTimeSelected;
				dr[TABLE.OrderedUnit.SelectedTechId]			= assigningTech;
				dr[TABLE.OrderedUnit.DivisionCode]			= _patientUnitSelection.OrderedComponent.DivisionCode;
				dr[TABLE.OrderedUnit.RecordStatusCode]		= Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
				dr[TABLE.OrderedUnit.LastUpdateUser]		= Common.LogonUser.LogonUserName;
				dr[TABLE.OrderedUnit.LastUpdateDate]		= VBECSDateTime.GetDivisionCurrentDateTime();
				// ToDo: Add code here if they change this...
				dr[TABLE.OrderedUnit.CrossmatchRequiredIndicator] = _patientUnitSelection._orderedComponent.SpecimenRequiredIndicator;

				_dtOrderedUnits.Rows.Add(dr);

				// must add any comments associated with this unit
				AddOrderedUnitComments( (Guid) dr[TABLE.OrderedUnit.OrderedUnitGuid] );
			}

			/// <summary>
			/// When a unit is removed from SelectedUnits, a corresponding OrderedUnit is also 
			/// removed.  
			/// </summary>
			/// <param name="bloodUnitGUID"></param>
			private void RemoveOrderedUnit(Guid bloodUnitGUID)
			{
				DataRow [] dr = this._dtOrderedUnits.Select(TABLE.OrderedUnit.BloodUnitGuid + " = '" + bloodUnitGUID + "'");
				if (dr.Length == 1)
				{
					Guid orderedUnitGUID = (Guid) dr[0][TABLE.OrderedUnit.OrderedUnitGuid];
					_dtOrderedUnits.Rows.Remove(dr[0]);
					// must also remove any comments associated with this unit
					RemoveOrderedUnitComments(orderedUnitGUID);
				}
			}

			/// <summary>
			/// Any comments associated with the OrderedUnit are added by this method when
			/// it is added.
			/// </summary>
			/// <param name="orderedUnitGUID"></param>
			private void AddOrderedUnitComments(Guid orderedUnitGUID)
			{
				foreach (DataRow drTemp in _dtTempOrderedUnitComments.Rows)
				{
					DataRow dr = _dtOrderedUnitComments.NewRow();
					dr.ItemArray = drTemp.ItemArray;
					dr[TABLE.OrderedUnit.OrderedUnitGuid] = orderedUnitGUID;
					dr[TABLE.OrderedUnitComment.OrderedUnitCommentGuid] = System.Guid.NewGuid();
					_dtOrderedUnitComments.Rows.Add(dr);
				}
			}

			/// <summary>
			/// Any comments associated with the OrderedUnit are removed by this method when
			/// it is removed.
			/// </summary>
			/// <param name="orderedUnitGUID"></param>
			private void RemoveOrderedUnitComments(Guid orderedUnitGUID)
			{
				DataRow [] drArray = this._dtOrderedUnitComments.Select(TABLE.OrderedUnit.OrderedUnitGuid + " = '" + orderedUnitGUID + "'");
				foreach (DataRow dr in drArray)
				{
					_dtOrderedUnitComments.Rows.Remove(dr);	
				}
			} 

			#endregion


			#region Implementation of ICollection

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/14/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7621"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7622"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This copies the collection to an array (not implemented).
			/// </summary>
			/// <param name="array"></param>
			/// <param name="index"></param>
			public void CopyTo(System.Array array, int index)
			{
				throw new BOL.BusinessObjectException("This method has not been implemented!");
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/14/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7623"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7624"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This indicates whether or not the collection is synchronized.  This is not implemented
			/// and is not needed since a user can only access one ordered component at a time.   Hmmm... So why is it here then?? 
			/// </summary>
			public bool IsSynchronized
			{
				get
				{
					return false;
				}
			}
            
			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/20/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3492"> 
			///		<ExpectedInput>None</ExpectedInput>
			///		<ExpectedOutput>1</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3493"> 
			///		<ExpectedInput>N/A</ExpectedInput>
			///		<ExpectedOutput>N/A</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			/// This returns the number of blood units associated with an order.
			///</summary>
			public int Count
			{
				get
				{
					return _bloodUnitsSelected.Count;
				}
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/14/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7625"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7626"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This has not been implemented.
			/// </summary>
			public object SyncRoot
			{
				get
				{
					throw new BOL.BusinessObjectException("This method has not been implemented!");
				}
			}
			#endregion
            

			#region Implementation of IEnumerable

			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/20/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="3494"> 
			///		<ExpectedInput>None</ExpectedInput>
			///		<ExpectedOutput>Valid instance of an enumerator</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="3495"> 
			///		<ExpectedInput>N/A</ExpectedInput>
			///		<ExpectedOutput>N/A</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// GetEnumerator
			/// </summary>
			/// <returns></returns>
			public System.Collections.IEnumerator GetEnumerator()
			{
				return new SelectedUnitEnumerator(this);
			}
			#endregion

			///<Developers>
			///	<Developer>Carl Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>10/21/2003</CreationDate>
			///
			///<summary>
			///This class in an implementation of the enumerator needed for the SelectedUnits 
			///class.
			///</summary>
			///
			public class SelectedUnitEnumerator : IEnumerator
			{
				private SelectedUnits _selectedUnits;	// A copy of the collection
				private int _index;						// track the index of the current element
				private Object currentElement;			// the current element, returned by Current

				internal SelectedUnitEnumerator(SelectedUnits selectedUnits) 
				{
					_selectedUnits = selectedUnits;
					_index = 0;
					currentElement = _selectedUnits._bloodUnitsSelected;

					if (_selectedUnits._bloodUnitsSelected.Count == 0) _index = -1;
				}


				///<Developers>
				///	<Developer>Carl Jensen</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>9/29/2003</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="3478"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="3479"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				///<summary>
				/// Implementation of MoveNext
				/// </summary>
				public virtual bool MoveNext() 
				{
					if (_index < 0) 
					{  // check to see if you can take this action. False indicates
						// that the MoveNext did not work
						currentElement = _selectedUnits._bloodUnitsSelected;
						return false;
					}
					
					
					IDictionaryEnumerator enumerator = this._selectedUnits._bloodUnitsSelected.GetEnumerator();
					int placeHolder = 0;
					while (enumerator.MoveNext())
					{
						if (placeHolder == this._index)
						{
							currentElement = enumerator.Value; // moved so reset current element
							_index++;                          // increase our index appropriately
							placeHolder++;
							
							//once at the end of our collection, ensure the user cannot MoveNext
							if (_index == _selectedUnits._bloodUnitsSelected.Count) 
							{
								_index = -1;
							}
							return true;
						}
						else
						{
							placeHolder++;
						}
					}
					return false;
				}

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/14/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7627"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7628"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// This is not implemented.
				/// </summary>
				public virtual void Reset() // resets the enumerator
				{
					throw new BOL.BusinessObjectException("This method has not been implemented!");
				}                 

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/14/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7629"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7630"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// This returns the current object.
				/// </summary>
				public virtual Object Current 
				{
					get 
					{
						if (currentElement == _selectedUnits._bloodUnitsSelected)
						{
							if (_index == 0)
								throw new InvalidOperationException("Invalid Operation");
							else
								throw new InvalidOperationException("Invalid operation");
						}
						return currentElement;
					}
				}
			}
		}

		#endregion
	}
}
